<?php

namespace App\Http\Controllers;

use App\Constants\Status;
use App\Models\Booking;
use App\Models\Review;
use App\Rules\Captcha;
use App\Models\Wishlist;
use App\Models\JobRequest;
use Hash, Image, File, Str;
use Illuminate\Http\Request;
use App\Models\ListingReport;
use Illuminate\Support\Facades\Auth;
use Modules\Listing\Entities\Listing;
use Modules\Kyc\Entities\KycInformation;
use Modules\Listing\Entities\ListingGallery;
use Modules\Listing\Entities\ListingTranslation;
use Modules\Subscription\Entities\SubscriptionHistory;

class ProfileController extends Controller
{

    public function __construct()
    {
        $this->middleware('auth:web');
    }

    public function dashboard(Request $request)
    {
        $user = Auth::guard('web')->user();

        $total_listing = Listing::where('agent_id', $user->id)->count();
        $pending_listing = Listing::where('agent_id', $user->id)->where('approved_by_admin', 'pending')->count();

        $total_featured_listing = Listing::where('agent_id', $user->id)->where('is_featured', 'enable')->count();

        $total_wishlist = Wishlist::where('user_id', $user->id)->count();

        $job_posts = Listing::where(['agent_id' => $user->id, 'category_id' => 3])->latest()->take(10)->get();


        return view('profile.dashboard', ['user' => $user, 'job_posts' => $job_posts, 'total_listing' => $total_listing, 'total_featured_listing' => $total_featured_listing, 'pending_listing' => $pending_listing, 'total_wishlist' => $total_wishlist]);
    }

    public function edit(Request $request)
    {
        $user = Auth::guard('web')->user();

        return view('profile.edit', [
            'user' => $user,
        ]);
    }

    public function update(Request $request)
    {

        $rules = [
            'name'=>'required',
            'email'=>'required',
            'phone'=>'required',
            'address'=>'required|max:220',
        ];
        $customMessages = [
            'name.required' => trans('translate.Name is required'),
            'email.required' => trans('translate.Email is required'),
            'phone.required' => trans('translate.Phone is required'),
            'address.required' => trans('translate.Address is required')
        ];
        $this->validate($request, $rules,$customMessages);

        $user = Auth::guard('web')->user();
        $user->name = $request->name;
        $user->phone = $request->phone;
        $user->whatsapp_phone = $request->whatsapp_phone;
        $user->address = $request->address;
        $user->designation = $request->designation;
        $user->google_map = $request->google_map;
        $user->about_me = $request->about_me;
        $user->instagram = $request->instagram;
        $user->facebook = $request->facebook;
        $user->linkedin = $request->linkedin;
        $user->twitter = $request->twitter;
        $user->sunday = $request->sunday;
        $user->monday = $request->monday;
        $user->tuesday = $request->tuesday;
        $user->wednesday = $request->wednesday;
        $user->thursday = $request->thursday;
        $user->friday = $request->friday;
        $user->saturday = $request->saturday;
        $user->save();

        if($request->file('image')){
            $old_image = $user->image;
            $user_image = $request->image;
            $extention = $user_image->getClientOriginalExtension();
            $image_name = Str::slug($user->name).date('-Y-m-d-h-i-s-').rand(999,9999).'.'.$extention;
            $image_name = 'uploads/custom-images/'.$image_name;
            Image::make($user_image)->save(public_path().'/'.$image_name);
            $user->image = $image_name;
            $user->save();
            if($old_image){
                if(File::exists(public_path().'/'.$old_image))unlink(public_path().'/'.$old_image);
            }
        }

        if($request->file('resume')){
            $file = $request->resume;
            $old_file = $user->resume;
            $file_ext = $file->getClientOriginalExtension();
            $file_name= 'resume-'.date('Y-m-d-h-i-s-').rand(999,9999).'.'.$file_ext;
            $file_path = $file_name;
            $file->move(public_path().'/uploads/custom-images/',$file_path);
            $user->resume = $file_path;
            $user->save();

            if($old_file){
                if(File::exists(public_path().'/'."uploads/custom-images/".$old_file)) unlink(public_path().'/'."uploads/custom-images/".$old_file);
            }
        }


        $notification= trans('translate.Your profile updated successfully');
        $notification=array('messege'=>$notification,'alert-type'=>'success');
        return redirect()->back()->with($notification);
    }

    public function change_password(Request $request)
    {
        return view('profile.change_password');
    }

    public function update_password(Request $request)
    {
        $rules = [
            'current_password'=>'required',
            'password'=>'required|min:4|confirmed',
        ];
        $customMessages = [
            'current_password.required' => trans('translate.Current password is required'),
            'password.required' => trans('translate.Password is required'),
            'password.min' => trans('translate.Password minimum 4 character'),
            'password.confirmed' => trans('translate.Confirm password does not match'),
        ];
        $this->validate($request, $rules,$customMessages);

        $user = Auth::guard('web')->user();
        if(Hash::check($request->current_password, $user->password)){
            $user->password = Hash::make($request->password);
            $user->save();

            $notification = trans('translate.Password change successfully');
            $notification=array('messege'=>$notification,'alert-type'=>'success');
            return redirect()->back()->with($notification);

        }else{
            $notification = trans('translate.Current password does not match');
            $notification=array('messege'=>$notification,'alert-type'=>'error');
            return redirect()->back()->with($notification);
        }
    }

    public function orders(){

        $user = Auth::guard('web')->user();

        $histories = SubscriptionHistory::where('user_id', $user->id)->latest()->paginate(10);

        return view('profile.orders', ['histories' => $histories, 'user' => $user]);
    }


    public function reviews(){

        $user = Auth::guard('web')->user();

        $reviews = Review::with('listing.agent')->latest()->where('user_id', $user->id)->latest()->paginate(10);

        return view('profile.reviews', ['reviews' => $reviews]);
    }

    public function store_review(Request $request){

        $rules = [
            'rating'=>'required',
            'comment'=>'required',
            'agent_id'=>'required',
            'listing_id'=>'required',
            'g-recaptcha-response'=>new Captcha()
        ];

        $customMessages = [
            'rating.required' => trans('translate.Rating is required'),
            'comment.required' => trans('translate.Review is required'),
        ];

        $this->validate($request, $rules,$customMessages);

        $user = Auth::guard('web')->user();

        if($request->agent_id == $user->id){
            $notification = trans('translate.You can not submit review to your own listing');
            $notification=array('messege'=>$notification,'alert-type'=>'error');
            return redirect()->back()->with($notification);
        }

        $is_exist = Review::where(['user_id' => $user->id, 'listing_id' => $request->listing_id])->count();

        if($is_exist == 0){
            $review = new Review();
            $review->user_id = $user->id;
            $review->rating = $request->rating;
            $review->comment = $request->comment;
            $review->agent_id = $request->agent_id;
            $review->listing_id = $request->listing_id;
            $review->save();

            $notification = trans('translate.Review submited successfully, review will be published after admin approval');
            $notification=array('messege'=>$notification,'alert-type'=>'success');
            return redirect()->back()->with($notification);

        }else{
            $notification = trans('translate.Review already submited');
            $notification=array('messege'=>$notification,'alert-type'=>'error');
            return redirect()->back()->with($notification);
        }

    }


    public function add_to_wishlist($id){
        $user = Auth::guard('web')->user();
        $is_exist = Wishlist::where(['user_id' => $user->id, 'listing_id' => $id])->count();
        if($is_exist == 0){

            $wishlist = new Wishlist();
            $wishlist->listing_id = $id;
            $wishlist->user_id = $user->id;
            $wishlist->save();

            $notification = trans('translate.Item added to wishlist');
            $notification=array('messege'=>$notification,'alert-type'=>'success');
            return redirect()->back()->with($notification);

        }else{
            $notification = trans('translate.Already added to wishlist');
            $notification=array('messege'=>$notification,'alert-type'=>'error');
            return redirect()->back()->with($notification);
        }

    }

    public function wishlists(){

        $user = Auth::guard('web')->user();
        $wishlists = Wishlist::where(['user_id' => $user->id])->get();
        $wishlist_arr = array();
        foreach($wishlists as $wishlist){
            $wishlist_arr [] = $wishlist->listing_id;
        }

        $listings = Listing::with('agent', 'sub_category')->where(function ($query) {
            $query->where('expired_date', null)
                ->orWhere('expired_date', '>=', date('Y-m-d'));
        })->where(['status' => 'enable', 'approved_by_admin' => 'approved'])->whereIn('id', $wishlist_arr)->get();


        return view('profile.wishlists', ['listings' => $listings]);

    }

    public function remove_wishlist($id){

        $user = Auth::guard('web')->user();
        Wishlist::where(['user_id' => $user->id, 'listing_id' => $id])->delete();

        $notification = trans('translate.Item remove to favourite list');
        $notification = array('messege'=>$notification,'alert-type'=>'success');
        return redirect()->back()->with($notification);
    }


    public function job_requests(){
        $user = Auth::guard('web')->user();
        $job_requests = JobRequest::with('listing')->where('user_id', $user->id)->latest()->paginate(10);

        return view('profile.job_requests', ['job_requests' => $job_requests]);

    }

    public function job_posts(){
        $user = Auth::guard('web')->user();

        $job_posts = Listing::where(['agent_id' => $user->id, 'category_id' => 3])->latest()->paginate(10);

        return view('profile.job_posts', ['job_posts' => $job_posts]);

    }

    public function job_post_applicants($id){

        $user = Auth::guard('web')->user();

        $job_post = Listing::where(['agent_id' => $user->id, 'id' => $id])->firstOrFail();

        $job_requests = JobRequest::with('user')->where('listing_id', $id)->latest()->paginate(10);

        return view('profile.job_applicants', ['job_requests' => $job_requests]);

    }

    public function job_application_approval($id){

        $job_request = JobRequest::findOrFail($id);

        $approval_check = JobRequest::where('listing_id', $job_request->listing_id)->where('status', 'approved')->count();

        if($approval_check == 0){
            $job_request = JobRequest::findOrFail($id);
            $job_request->status = 'approved';
            $job_request->save();

            JobRequest::where('listing_id', $job_request->listing_id)->where('id', '!=', $id)->update(['status' => 'rejected']);

            $notification = trans('translate.Job assigned successfully');
            $notification = array('messege'=>$notification,'alert-type'=>'success');
            return redirect()->back()->with($notification);

        }else{
            $notification = trans('translate.Job already has assigned, so you can not assign again');
            $notification = array('messege'=>$notification,'alert-type'=>'error');
            return redirect()->back()->with($notification);
        }

    }

    public function delete_account(Request $request){

        return view('profile.delete_account');
    }

    public function submit_delete_account(Request $request){

        $rules = [
            'current_password'=>'required',
        ];
        $customMessages = [
            'current_password.required' => trans('translate.Current password is required'),
        ];
        $this->validate($request, $rules,$customMessages);

        $user = Auth::guard('web')->user();
        if(Hash::check($request->current_password, $user->password)){


            $user_image = $user->image;

            if($user_image){
                if(File::exists(public_path().'/'.$user_image))unlink(public_path().'/'.$user_image);
            }

            $id = $user->id;

            Review::where('user_id',$id)->delete();
            SubscriptionHistory::where('user_id',$id)->delete();
            Wishlist::where('user_id',$id)->delete();
            JobRequest::where('user_id',$id)->delete();
            JobRequest::where('agent_id',$id)->delete();
            ListingReport::where('agent_id',$id)->delete();

            $kyc = KycInformation::where(['user_id' => $id])->first();

            if($kyc){
                $kyc_file = $kyc->file;
                if($kyc_file){
                    if(File::exists(public_path().'/'.$kyc_file))unlink(public_path().'/'.$kyc_file);
                }
            }

            $kyc->delete();


            $listings = Listing::where('agent_id', $user->id)->latest()->get();

            foreach($listings as $listing){

                $old_image = $listing->thumb_image;

                if($old_image){
                    if(File::exists(public_path().'/'.$old_image))unlink(public_path().'/'.$old_image);
                }

                ListingTranslation::where('listing_id',$id)->delete();
                Review::where('listing_id',$id)->delete();
                ListingReport::where('listing_id',$id)->delete();
                JobRequest::where('listing_id',$id)->delete();

                $galleries = ListingGallery::where('listing_id', $id)->get();
                foreach($galleries as $gallery){
                    $old_image = $gallery->image;

                    if($old_image){
                        if(File::exists(public_path().'/'.$old_image))unlink(public_path().'/'.$old_image);
                    }

                    $gallery->delete();
                }

                $listing->delete();

            }

            $user->delete();

            Auth::guard('web')->logout();

            $notification = trans('translate.Your account has been deleted successfully');
            $notification=array('messege'=>$notification,'alert-type'=>'success');
            return redirect()->route('home')->with($notification);

        }else{
            $notification = trans('translate.Current password does not match');
            $notification=array('messege'=>$notification,'alert-type'=>'error');
            return redirect()->back()->with($notification);
        }

    }


    public function bookListing(Request $request)
    {
        $request->validate([
            'listing_id' => 'required|exists:listings,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
        ]);

        $userId = auth()->guard('web')->user()->id;
        $existingBooking = Booking::where('listing_id', $request->input('listing_id'))
            ->where('user_id', $userId)
            ->where(function ($query) use ($request) {
                $query->whereDate('start_date', $request->input('start_date'))
                    ->orWhereDate('end_date', $request->input('end_date'));
            })
            ->first();

        if ($existingBooking) {
            $notification = trans('translate.You have already booked this listing for the selected date');
            $notification = array('messege' => $notification, 'alert-type' => 'error');
            return redirect()->back()->with($notification);
        }

        $booking = new Booking();
        $booking->listing_id = $request->input('listing_id');
        $booking->user_id = $userId;
        $booking->start_date = $request->input('start_date');
        $booking->end_date = $request->input('end_date');
        $booking->status = Status::ENABLE;
        $booking->save();

        $notification = trans('translate.Booking created successfully');
        $notification = array('messege' => $notification, 'alert-type' => 'success');
        return redirect()->back()->with($notification);
    }


    public function myBookingList()
    {
        $user = Auth::guard('web')->user();
        $bookings = Booking::with('listing')->where('user_id', $user->id)->latest()->paginate(10);

        return view('profile.booking_list', compact('bookings'));
    }

}
